local fluid_names = {
    { id = "gtceu:oil_medium",     name = "Raw Oil" },
    { id = "gtceu:naphtha",        name = "Naphtha" },
    { id = "gtceu:jean_gasoline",  name = "Jean Gas" },
    { id = "gtceu:liquid_oxygen",  name = "Liquid Oxygen" },
    { id = "gtceu:oxygen",         name = "Oxygen" },
    { id = "gtceu:hydrogen",       name = "Hydrogen" },
    { id = "gtceu:nitrogen",       name = "Nitrogen" },
    { id = "gtceu:chlorine",       name = "Chlorine" }
}

local monitor = peripheral.wrap("back")
local storage = peripheral.wrap("bottom")

monitor.setTextScale(1.5)
monitor.clear()

local function fmt_buckets(mb)
    local b = mb / 1000
    if b >= 1e9 then
        return string.format("%.1f GB", b / 1e9)
    elseif b >= 1e6 then
        return string.format("%.1f MB", b / 1e6)
    elseif b >= 1e3 then
        return string.format("%.1f KB", b / 1e3)
    else
        return string.format("%.1f B", b)
    end
end

local function get_fluid_map()
    local fluid_list = storage.listFluid()
    local map = {}
    for _, fluid in ipairs(fluid_list) do
        map[fluid.name] = fluid.amount
    end
    return map
end

local function center(text, length)
    local pad = length - #text
    if pad <= 0 then return text end
    local left = math.floor(pad / 2)
    local right = pad - left
    return string.rep(" ", left) .. text .. string.rep(" ", right)
end

local function ct(text, monitor)
    local width = select(1, monitor.getSize())
    local pad = width - #text
    if pad <= 0 then return text end
    local left = math.floor(pad / 2)
    local right = pad - left
    return string.rep(" ", left) .. text .. string.rep(" ", right)
end

local function display(ratios, current_amounts)
    monitor.clear()
    monitor.setCursorPos(1, 1)
    monitor.setTextColor(colors.white)
    monitor.write(ct(string.format("%s %s %s", center("Ratio/m", 11), center("Ratio/s", 16), center("Storage", 11)), monitor))

    for _, entry in ipairs(fluid_names) do
        local id = entry.id
        local label = entry.name
        local ratio_per_s = ratios[id] or 0
        local ratio_per_m = ratio_per_s * 60
        local current = current_amounts[id] or 0

        monitor.setCursorPos(1, select(2, monitor.getCursorPos()) + 1)
        monitor.write(ct(string.format("=========== %s ===========", center(label, 16)), monitor))

        local r_text = string.format("%+6.1f B/m ", ratio_per_m / 1000)
        local d_text = string.format("%+6.1f B/s ", ratio_per_s / 1000)
        local c_text = fmt_buckets(current)

        monitor.setCursorPos(1, select(2, monitor.getCursorPos()) + 1)

        monitor.setTextColor(ratio_per_m >= 0 and colors.lime or colors.red)
        monitor.write(center(r_text, 7))

        monitor.setTextColor(ratio_per_s >= 0 and colors.lime or colors.red)
        monitor.write(center(d_text, 17))

        monitor.setTextColor(colors.blue)
        monitor.write(center(c_text, 13))
        monitor.setTextColor(colors.white)
    end
end

local previous = get_fluid_map()
local history = {}

for _, entry in ipairs(fluid_names) do
    local id = entry.id
    history[id] = {}
    if previous[id] == nil then previous[id] = 0 end
end

while true do
    local current = get_fluid_map()
    local ratios = {}

    for _, entry in ipairs(fluid_names) do
        local id = entry.id
        local curr = current[id] or 0
        local prev = previous[id] or curr
        local diff = curr - prev

        local h = history[id]
        table.insert(h, diff)
        if #h > 60 then table.remove(h, 1) end

        local sum = 0
        for _, v in ipairs(h) do sum = sum + v end
        ratios[id] = sum / #h

        previous[id] = curr
    end

    display(ratios, current)
    sleep(1)
end
